/** \file
 * SmartKey Driver Installation.
 *
 * The suggested way to install the SmartKey drivers is to simply call only the
 * required SkeyInstallPar(), SkeyInstallUSB(), SkeyInstallGSS2() and SkeyInstallVDD() 
 * functions.
 *
 * If a function fails with the error code SKEYINST_ERROR_WAIT you need
 * to ask at the user to close any other installation process and retry.
 * If a function fails with the error code SKEYINST_ERROR_ADMIN 
 * you need to ask at the user to became Administrator and restart the installation.
 * If a function fails with any other error code the installation is failed.
 *
 * If a function returns with the code SKEYINST_OK, the operation is completed 
 * with success.
 * If a function returns with the code SKEYINST_WARNING_MUST_REBOOT, you must
 * reboot the system to complete the operation.
 * If a function returns with the code SKEYINST_WARNING_MUST_INSERT, you need
 * to ask at the used to insert the SmartKey to complete the process.
 *
 * Please note that you don't need to check if a driver is already installed
 * before calling the install functions. If the driver is already installed and
 * updated these functions do nothing.
 * 
 */

#ifndef __SKEYINST_H
#define __SKEYINST_H

#ifdef __cplusplus
extern "C" {
#endif

/* Only for internal use */
#ifdef SKEYINST_EXPORTS
#define SKEYINST_LINK
#else
#define SKEYINST_LINK __declspec(dllimport)
#endif
#define SKEYINST_CALL __stdcall

/**
 * Installation of the USB driver.
 * All the required drivers for the USB SmartKey are installed.
 * On Windows NT the USB Phoenix stack is also installed.
 * \note
 * The files skeyusb.inf, skeyusbnt.inf, skeyusb.sys, skeyusbnt.sys, usbdrv.sys,
 * eusk3usb.sys, eusk3usb.inf and eusk3usb.cat must reside in the same directory of the DLL.
 * To support 64 bits systems you must add also eusk3usb.*.inf eusk3usb.*.sys and euinst.*.exe files.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyInstallUSB(void);

/**
 * Installation of the Parallel driver.
 * All the required drivers for the Parallel SmartKey are installed.
 * If in the current operating system the driver isn't needed, nothing is done and the function returns with success.
 * \note
 * The driver is started as "Automatic". If you need to use the SmartKey in a Windows Service, you must add a 
 * dependency on the eusk2par driver, to prevent the service to start before the SmartKey driver.
 * \note
 * The file eusk2par.sys must reside in the same directory of the DLL.
 * To support 64 bits systems you must add also eusk2par.*.sys files. 
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyInstallPar(void);

/**
 * Installation of the GSS2 driver.
 * All the required drivers for the GSS2 SmartKey are installed.
 * If in the current operating system the driver isn't needed, nothing is done and the function returns with success.
 * \note
 * The files eugss*.sys and psapi.dll must reside in the same directory of the DLL.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyInstallGSS2(void);

/**
 * Installation of the VDD driver.
 * All the required drivers for 16 bits applications are installed. In Windows Vista
 * the VDD driver must be installed in the Windows System directory. It cannot resides 
 * in the same directory of the executable.
 * If in the current operating system the driver isn't needed, nothing is done and the function returns with success.
 * \note
 * The file skeyvdd.dll must reside in the same directory of the DLL.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyInstallVDD(void);

/**
 * Uninstallation of the USB driver.
 * The installed USB drivers are uninstalled. 
 * If no driver is installed nothing is done and the function returns with success.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyUnInstallUSB(void);

/**
 * Uninstallation of the Parallel driver.
 * The installed Parallel drivers are uninstalled.
 * If no driver is installed nothing is done and the function returns with success.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyUnInstallPar(void);

/**
 * Uninstallation of the GSS2 driver.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyUnInstallGSS2(void);

/**
 * Uninstallation of the VDD driver.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyUnInstallVDD(void);

/**
 * Forced uninstallation of the USB driver.
 * The driver is uninstalled with force removing any reference on the filesystem and registry.
 * \note Use only in extreme cases to recover from wrong installation.
 * \note After a forced uninstallation you MUST always reboot.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyForceUnInstallUSB(void);

/**
 * Forced uninstallation of the Parallel driver.
 * The driver is uninstalled with force removing any reference on the filesystem and registry.
 * \note Use only in extreme cases to recover from wrong installation.
 * \note After a forced uninstallation you MUST always reboot.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyForceUnInstallPar(void);

/**
 * Forced uninstallation of the GSS2 driver.
 * The driver is uninstalled with force removing any reference on the filesystem and registry. 
 * \note Use only in extreme cases to recover from wrong installation.
 * \note After a forced uninstallation you MUST always reboot.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyForceUnInstallGSS2(void);

/**
 * Forced uninstallation of the VDD driver.
 * The driver is uninstalled with force removing any reference on the filesystem and registry. 
 * \note Use only in extreme cases to recover from wrong installation.
 * \note After a forced uninstallation you MUST always reboot.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyForceUnInstallVDD(void);

/**
 * Get the last Windows error code.
 * \return the last Windows error code like the standard Windows GetLastError() function.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyGetLastError(void);

/**
 * Select the log file to use.
 * If you specify a log file, when you call one of the install or uninstall functions
 * a very detailed report of all the operations is generated. 
 * This report can be used to debug problematic installation cases.
 * \param Path Complete path of the log file. Use NULL to disable it.
 * \return A SKEYINST_* code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyLogFile(const char* Path);

#ifdef SKEYINST_MSI

/**
 * Install all the SmartKey drivers. 
 * Like a sequence call of SkeyInstallUSB(), SkeyInstallPar(), SkeyInstallGSS2() and SkeyInstallVDD().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAInstallAll(MSIHANDLE hInstall);

/**
 * Uninstall all the SmartKey drivers. 
 * Like a sequence call of SkeyUnInstallUSB(), SkeyUnInstallPar(), SkeyUnInstallGSS2() and SkeyUnInstallVDD().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAUninstallAll(MSIHANDLE hInstall);

/**
 * Installation of the USB driver.
 * Like SkeyInstallUSB().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAInstallUSB(MSIHANDLE hInstall);

/**
 * Uninstallation of the USB driver.
 * Like SkeyUnInstallUSB().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAUninstallUSB(MSIHANDLE hInstall);

/**
 * Installation of the Parallel driver.
 * Like SkeyInstallPar().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAInstallPar(MSIHANDLE hInstall);

/**
 * Uninstallation of the Parallel drive.
 * Like SkeyUnInstallParallel().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAUninstallPar(MSIHANDLE hInstall);

/**
 * Installation of the GSS2 driver.
 * Like SkeyInstallGSS2().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAInstallGSS2(MSIHANDLE hInstall);

/**
 * Uninstallation of the GSS2 driver.
 * Like SkeyUnInstallGSS2().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAUninstallGSS2(MSIHANDLE hInstall);

/**
 * Installation of the VDD driver.
 * Like SkeyInstallVDD().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAInstallVDD(MSIHANDLE hInstall);

/**
 * Uninstallation of the VDD driver.
 * Like SkeyUnInstallVDD().
 * \note This function is to be used from inside an MSI installation via a custom action calls.
 * \param hInstall The MSI database handle.
 * \return ERROR_SUCCESS on success or another error code.
 */
SKEYINST_LINK DWORD SKEYINST_CALL SkeyCAUninstallVDD(MSIHANDLE hInstall);
#endif

/** \name Error
 * Error codes returned.
 */
/*@{*/

/**
 * You don't have Administrator rights.
 *
 * To install drivers you must be an Administrator of the system.
 */
#define SKEYINST_ERROR_ADMIN -4 

/** 
 * Another installation is in progress. 
 * You must ask at the user to complete any installation process in progress and then retry.
 *
 * A good behavior is:
 *  - Try the first time
 *  - Wait 5 seconds and retry
 *  - Ask the user to complete and to close any installation window and retry
 *  - Repeat the last step
 *
 * Note that the Windows XP tooltip, shown on the lower right corner of the screen 
 * after an hardware installation, must be closed because is part of the installation 
 * process.
 */
#define SKEYINST_ERROR_WAIT -3 

/** 
 * This operation is unsupported in this operating system.
 *
 * You can't call the function in this operation system.
 */
#define SKEYINST_ERROR_UNSUPPORTED -2 

/**
 * Generic failure.
 *
 * The operation is failed for a generic error. You can call the SkeyGetLastError() function 
 * to get the Windows error code. 
 */
#define SKEYINST_ERROR_FAIL -1 

/**
 * Operation completed successfully.
 */
#define SKEYINST_OK 0

/**
 * Operation completed successfully, but you must reboot to complete the operation.
 * This code is generally returned after the forced uninstallation.
 */
#define SKEYINST_WARNING_MUST_REBOOT 1

/**
 * Operation completed successfully, but you must insert the token to complete the operation.
 * This code is generally returned after the installation of the USB Driver on Windows 9x/Me/2000.
 *
 * The insertion is required to install the driver for the token. 
 * At the time of the insertion you must have Administrator rights.
 */
#define SKEYINST_WARNING_MUST_INSERT 2

/*@}*/

#ifdef __cplusplus
}
#endif

#endif